-- ========================================
-- SOLUCIÓN DEFINITIVA: TRÁMITES DERIVADOS
-- ========================================

-- PROBLEMAS ENCONTRADOS:
-- 1. El primer movimiento (registro) queda con estado VACÍO
-- 2. Los movimientos de derivación quedan con estado PENDIENTE (deberían ser DERIVADO)
-- 3. El area_origen_id del tercer movimiento es incorrecto (toma área incorrecta)

-- ========================================
-- PASO 1: Corregir SP_REGISTRAR_TRAMITE_DERIVAR
-- ========================================

DROP PROCEDURE IF EXISTS SP_REGISTRAR_TRAMITE_DERIVAR;

DELIMITER $$

CREATE PROCEDURE SP_REGISTRAR_TRAMITE_DERIVAR(
    IN IDDO VARCHAR(50),
    IN ORIG INT,           -- Área del usuario que está derivando
    IN DEST INT,           -- Área de destino
    IN DESCRIP VARCHAR(255),
    IN IDUSU INT,
    IN RUTA VARCHAR(255),
    IN ACCION VARCHAR(20), -- 'DERIVAR' o 'FINALIZAR'
    IN VISIBLE_PUBLICO TINYINT
)
BEGIN
    DECLARE ESTATUS_NUEVO VARCHAR(20);

    -- Determinar el estatus del NUEVO movimiento
    IF ACCION = 'DERIVAR' THEN
        SET ESTATUS_NUEVO = 'DERIVADO';     -- El nuevo movimiento es DERIVADO
    ELSE
        SET ESTATUS_NUEVO = 'FINALIZADO';   -- El nuevo movimiento es FINALIZADO
    END IF;

    -- NO ACTUALIZAMOS el movimiento anterior
    -- Se queda como está (PENDIENTE)

    -- Insertar el NUEVO movimiento con estado DERIVADO
    INSERT INTO movimiento(
        documento_id,
        area_origen_id,     -- Área del usuario que deriva (ORIG)
        area_destino_id,    -- Área de destino (DEST)
        mov_descripcion,
        mov_estatus,        -- DERIVADO (el que deriva)
        mov_fecharegistro,
        usuario_id,
        mov_archivo,
        mov_visible_publico
    ) VALUES (
        IDDO,
        ORIG,                                         -- Quien deriva
        IF(DEST = 0 OR DEST IS NULL, NULL, DEST),   -- A quién deriva
        DESCRIP,
        ESTATUS_NUEVO,      -- DERIVADO
        NOW(),
        IDUSU,
        RUTA,
        VISIBLE_PUBLICO
    );

    -- PASO 3: Actualizar el documento
    UPDATE documento SET
        doc_estatus = IF(ACCION = 'FINALIZAR', 'FINALIZADO', doc_estatus)
    WHERE documento_id = IDDO;
END$$

DELIMITER ;

SELECT '✅ SP_REGISTRAR_TRAMITE_DERIVAR corregido' AS Resultado;

-- ========================================
-- PASO 2: SP_LISTAR_TRAMITE_AREA (Trámites Recibidos)
-- ========================================
-- Debe mostrar solo los trámites donde hay un movimiento PENDIENTE en mi área
-- Y ese movimiento debe tener un área_origen diferente a la mía

DROP PROCEDURE IF EXISTS SP_LISTAR_TRAMITE_AREA;

DELIMITER $$

CREATE PROCEDURE SP_LISTAR_TRAMITE_AREA(
    IN IDUSUARIO INT,
    IN TIPO_RE VARCHAR(20)
)
BEGIN
    DECLARE IDAREA INT;
    SET @IDAREA := (SELECT area_id FROM usuario WHERE usu_id = IDUSUARIO);

    SELECT
        SUBSTRING_INDEX(documento.documento_id, '-', 1) AS expe_num,
        documento.documento_id,
        documento.doc_nrodocumento,
        documento.doc_dniremitente,
        documento.doc_nombreremitente,
        documento.doc_apepatremitente,
        documento.doc_apematremitente,
        documento.doc_emailremitente,
        documento.doc_celularremitente,
        documento.doc_direccionremitente,
        documento.doc_representacion,
        documento.doc_ruc,
        documento.doc_empresa,
        documento.doc_folio,
        documento.doc_archivo,
        documento.doc_asunto,
        documento.doc_fecharegistro,
        documento.tipodocumento_id,
        documento.area_origen,
        documento.area_destino,
        documento.doc_estatus,
        tipo_documento.tipodocumento_id AS tipodoc_id,
        tipo_documento.tipodoc_descripcion,
        tupa.tupa_id,
        tupa.tupa_descripcion,
        tupa.tupa_requisitos,
        movimiento.movimiento_id,
        movimiento.area_origen_id,
        movimiento.area_destino_id,
        movimiento.mov_estatus,
        movimiento.mov_descripcion,
        movimiento.mov_archivo,
        movimiento.mov_fecharegistro,
        areaorigen.area_nombre AS origen,
        areadestino.area_nombre AS destino,
        documento.grupo_area_destino_id,
        grupo.area_nombre AS grupo,
        CONCAT_WS(' ',
            documento.doc_nombreremitente,
            documento.doc_apepatremitente,
            documento.doc_apematremitente
        ) AS REMITENTE
    FROM movimiento
    INNER JOIN documento ON movimiento.documento_id = documento.documento_id
    LEFT JOIN tipo_documento ON documento.tipodocumento_id = tipo_documento.tipodocumento_id
    LEFT JOIN tupa ON documento.tupa_id = tupa.tupa_id
    LEFT JOIN area AS areaorigen ON movimiento.area_origen_id = areaorigen.area_cod
    LEFT JOIN area AS areadestino ON movimiento.area_destino_id = areadestino.area_cod
    LEFT JOIN area AS grupo ON documento.grupo_area_destino_id = grupo.area_cod
    WHERE
        movimiento.area_destino_id = @IDAREA
        AND movimiento.mov_estatus = 'PENDIENTE'
        AND movimiento.movimiento_id = (
            SELECT MAX(m2.movimiento_id)
            FROM movimiento m2
            WHERE m2.documento_id = movimiento.documento_id
        )
        AND (TIPO_RE = '' OR documento.doc_representacion = TIPO_RE)
    ORDER BY movimiento.mov_fecharegistro DESC;
END$$

DELIMITER ;

SELECT '✅ SP_LISTAR_TRAMITE_AREA corregido' AS Resultado;

-- ========================================
-- PASO 3: SP_LISTAR_TRAMITE_AREA_DERIVADO (Trámites Derivados)
-- ========================================
-- Debe mostrar los trámites con estado DERIVADO donde YO soy el origen

DROP PROCEDURE IF EXISTS SP_LISTAR_TRAMITE_AREA_DERIVADO;

DELIMITER $$

CREATE PROCEDURE SP_LISTAR_TRAMITE_AREA_DERIVADO(
    IN IDUSUARIO INT
)
BEGIN
    DECLARE IDAREA INT;
    SET @IDAREA := (SELECT area_id FROM usuario WHERE usu_id = IDUSUARIO);

    SELECT
        SUBSTRING_INDEX(documento.documento_id, '-', 1) AS expe_num,
        documento.documento_id,
        documento.doc_nrodocumento,
        documento.doc_dniremitente,
        documento.doc_nombreremitente,
        documento.doc_apepatremitente,
        documento.doc_apematremitente,
        documento.doc_emailremitente,
        documento.doc_celularremitente,
        documento.doc_direccionremitente,
        documento.doc_representacion,
        documento.doc_ruc,
        documento.doc_empresa,
        documento.doc_folio,
        documento.doc_archivo,
        documento.doc_asunto,
        documento.doc_fecharegistro,
        documento.tipodocumento_id,
        documento.area_origen,
        documento.area_destino,
        documento.doc_estatus,
        tipo_documento.tipodocumento_id AS tipodoc_id,
        tipo_documento.tipodoc_descripcion,
        tupa.tupa_id,
        tupa.tupa_descripcion,
        tupa.tupa_requisitos,
        movimiento.movimiento_id,
        movimiento.area_origen_id,
        movimiento.area_destino_id,
        movimiento.mov_estatus,
        movimiento.mov_descripcion,
        movimiento.mov_archivo,
        movimiento.mov_fecharegistro,
        areaorigen.area_nombre AS origen,
        areadestino.area_nombre AS destino,
        documento.grupo_area_destino_id,
        grupo.area_nombre AS grupo,
        CONCAT_WS(' ',
            documento.doc_nombreremitente,
            documento.doc_apepatremitente,
            documento.doc_apematremitente
        ) AS REMITENTE
    FROM movimiento
    INNER JOIN documento ON movimiento.documento_id = documento.documento_id
    LEFT JOIN tipo_documento ON documento.tipodocumento_id = tipo_documento.tipodocumento_id
    LEFT JOIN tupa ON documento.tupa_id = tupa.tupa_id
    LEFT JOIN area AS areaorigen ON movimiento.area_origen_id = areaorigen.area_cod
    LEFT JOIN area AS areadestino ON movimiento.area_destino_id = areadestino.area_cod
    LEFT JOIN area AS grupo ON documento.grupo_area_destino_id = grupo.area_cod
    WHERE
        movimiento.area_origen_id = @IDAREA
        AND movimiento.mov_estatus = 'DERIVADO'
    ORDER BY movimiento.mov_fecharegistro DESC;
END$$

DELIMITER ;

SELECT '✅ SP_LISTAR_TRAMITE_AREA_DERIVADO corregido' AS Resultado;

-- ========================================
-- PASO 4: CORREGIR DATOS EXISTENTES
-- ========================================

-- Corregir el documento 001-2025 existente según lo explicado
UPDATE movimiento SET mov_estatus = 'PENDIENTE' WHERE movimiento_id = 1 AND documento_id = '001-2025';
UPDATE movimiento SET mov_estatus = 'DERIVADO' WHERE movimiento_id = 2 AND documento_id = '001-2025';
UPDATE movimiento SET mov_estatus = 'DERIVADO', area_origen_id = 29 WHERE movimiento_id = 3 AND documento_id = '001-2025';

SELECT '✅ Datos del documento 001-2025 corregidos' AS Resultado;

-- ========================================
-- VERIFICACIÓN
-- ========================================

SELECT '📊 Verificación de movimientos corregidos:' AS Info;

SELECT
    movimiento_id,
    area_origen_id,
    area_destino_id,
    mov_estatus,
    mov_fecharegistro
FROM movimiento
WHERE documento_id = '001-2025'
ORDER BY movimiento_id;

SELECT '✅ ¡TODO CORREGIDO!' AS Resultado;
SELECT 'Ahora prueba:' AS Info;
SELECT '- Gerencia debe ver el trámite en Derivados' AS Test1;
SELECT '- Informática debe ver el trámite en Derivados' AS Test2;
SELECT '- Mesa de Partes debe ver el trámite en Recibidos' AS Test3;
