<?php
session_start();

// Validar que se reciba el ID del movimiento
if (!isset($_GET['mov']) || empty($_GET['mov'])) {
    header("HTTP/1.0 400 Bad Request");
    die('Solicitud inválida');
}

// Validar que se reciba el DNI o código de verificación
if (!isset($_GET['cod']) || empty($_GET['cod'])) {
    header("HTTP/1.0 403 Forbidden");
    die('Acceso no autorizado');
}

require_once "model/model_conexion.php";

// Función para obtener datos del movimiento y validar acceso
function validarAccesoArchivo($movimiento_id, $codigo_verificacion) {
    try {
        $conexion = new ConexionDB();
        $c = $conexion->conexionPDO();

        // Obtener información del movimiento y validar acceso
        $sql = "SELECT
                    m.mov_archivo,
                    m.documento_id,
                    d.doc_dniremitente
                FROM movimiento m
                INNER JOIN documento d ON m.documento_id = d.documento_id
                WHERE m.movimiento_id = :mov_id
                AND m.mov_archivo IS NOT NULL
                AND m.mov_archivo != ''
                LIMIT 1";

        $query = $c->prepare($sql);
        $query->bindParam(':mov_id', $movimiento_id, PDO::PARAM_INT);
        $query->execute();

        $resultado = $query->fetch(PDO::FETCH_ASSOC);

        if (!$resultado) {
            return false;
        }

        // Validar que el código de verificación coincida con el DNI del remitente
        // Esto asegura que solo quien conoce el DNI pueda descargar
        if ($resultado['doc_dniremitente'] !== $codigo_verificacion) {
            return false;
        }

        return $resultado;

    } catch (PDOException $e) {
        error_log("Error en validarAccesoArchivo: " . $e->getMessage());
        return false;
    }
}

// Validar acceso
$movimiento_id = intval($_GET['mov']);
$codigo_verificacion = htmlspecialchars($_GET['cod'], ENT_QUOTES, 'UTF-8');

$datos = validarAccesoArchivo($movimiento_id, $codigo_verificacion);

if (!$datos) {
    header("HTTP/1.0 403 Forbidden");
    die('Acceso denegado. No tiene permisos para descargar este archivo.');
}

// Obtener la ruta del archivo
$ruta_archivo = $datos['mov_archivo'];
$ruta_completa = __DIR__ . '/' . $ruta_archivo;

// Validar que el archivo exista
if (!file_exists($ruta_completa)) {
    header("HTTP/1.0 404 Not Found");
    die('Archivo no encontrado');
}

// Validar que sea un archivo (no un directorio)
if (!is_file($ruta_completa)) {
    header("HTTP/1.0 403 Forbidden");
    die('Acceso denegado');
}

// Validar que el archivo esté dentro de las carpetas permitidas
$rutas_permitidas = [
    'controller/tramite/docs/',
    'controller/tramite_area/docs/'
];

$es_ruta_valida = false;
foreach ($rutas_permitidas as $ruta_permitida) {
    if (strpos($ruta_archivo, $ruta_permitida) === 0) {
        $es_ruta_valida = true;
        break;
    }
}

if (!$es_ruta_valida) {
    header("HTTP/1.0 403 Forbidden");
    die('Acceso denegado');
}

// Obtener el nombre del archivo
$nombre_archivo = basename($ruta_completa);

// Detectar el tipo MIME del archivo
$finfo = finfo_open(FILEINFO_MIME_TYPE);
$mime_type = finfo_file($finfo, $ruta_completa);
finfo_close($finfo);

// Tipos MIME permitidos
$mimes_permitidos = [
    'application/pdf',
    'application/msword',
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'image/jpeg',
    'image/png',
    'image/jpg'
];

if (!in_array($mime_type, $mimes_permitidos)) {
    header("HTTP/1.0 403 Forbidden");
    die('Tipo de archivo no permitido');
}

// OPCIONAL: Registrar la descarga en un log de auditoría
// $this->registrarDescarga($movimiento_id, $codigo_verificacion, $_SERVER['REMOTE_ADDR']);

// Limpiar el buffer de salida
if (ob_get_level()) {
    ob_end_clean();
}

// Configurar headers para la descarga
header('Content-Type: ' . $mime_type);
header('Content-Disposition: inline; filename="' . $nombre_archivo . '"');
header('Content-Length: ' . filesize($ruta_completa));
header('Cache-Control: private, max-age=0, must-revalidate');
header('Pragma: public');
header('Expires: 0');

// Prevenir ataques XSS con Content-Security-Policy
header("Content-Security-Policy: default-src 'none'; style-src 'unsafe-inline'; sandbox");
header("X-Content-Type-Options: nosniff");
header("X-Frame-Options: DENY");

// Leer y enviar el archivo en chunks para archivos grandes
$file = fopen($ruta_completa, 'rb');
while (!feof($file)) {
    echo fread($file, 8192);
    flush();
}
fclose($file);

exit;
?>
