-- ========================================
-- CORRECCIÓN: SP_REGISTRAR_TRAMITE_DERIVAR
-- ========================================
-- PROBLEMA: El movimiento anterior NO se actualiza a DERIVADO
-- SOLUCIÓN: Mejorar la lógica del UPDATE

DROP PROCEDURE IF EXISTS SP_REGISTRAR_TRAMITE_DERIVAR;

DELIMITER $$

CREATE PROCEDURE SP_REGISTRAR_TRAMITE_DERIVAR(
    IN IDDO VARCHAR(50),
    IN ORIG INT,
    IN DEST INT,
    IN DESCRIP VARCHAR(255),
    IN IDUSU INT,
    IN RUTA VARCHAR(255),
    IN ACCION VARCHAR(20),
    IN VISIBLE_PUBLICO TINYINT
)
BEGIN
    DECLARE ESTATUS VARCHAR(20);
    DECLARE MOVIMIENTOS_ACTUALIZADOS INT;

    -- Determinar el estatus del NUEVO movimiento
    IF ACCION = 'DERIVAR' THEN
        SET ESTATUS = 'PENDIENTE';  -- El nuevo movimiento queda PENDIENTE
    ELSE
        SET ESTATUS = 'FINALIZADO';  -- Si es FINALIZAR, el nuevo movimiento queda FINALIZADO
    END IF;

    -- PASO 1: Actualizar el movimiento ANTERIOR (el que estaba PENDIENTE) a DERIVADO o FINALIZADO
    -- CRÍTICO: Esto hace que desaparezca de "Trámites Recibidos"
    UPDATE movimiento
    SET mov_estatus = ACCION  -- Cambia PENDIENTE a DERIVADO o FINALIZADO
    WHERE documento_id = IDDO
    AND area_destino_id = ORIG  -- El movimiento que tenía como destino el área actual
    AND mov_estatus = 'PENDIENTE'  -- Solo actualizar los que están PENDIENTE
    ORDER BY movimiento_id DESC
    LIMIT 1;  -- Solo el último movimiento PENDIENTE

    -- Verificar cuántos registros se actualizaron
    SET MOVIMIENTOS_ACTUALIZADOS = ROW_COUNT();

    -- Si no se actualizó ninguno, buscar el último movimiento sin importar el estado
    -- (para casos donde el estado estaba vacío)
    IF MOVIMIENTOS_ACTUALIZADOS = 0 THEN
        UPDATE movimiento
        SET mov_estatus = ACCION
        WHERE documento_id = IDDO
        AND area_destino_id = ORIG
        AND (mov_estatus = '' OR mov_estatus IS NULL OR mov_estatus = 'PENDIENTE')
        ORDER BY movimiento_id DESC
        LIMIT 1;
    END IF;

    -- PASO 2: Insertar el NUEVO movimiento (derivación o finalización)
    INSERT INTO movimiento(
        documento_id,
        area_origen_id,
        area_destino_id,
        mov_descripcion,
        mov_estatus,
        mov_fecharegistro,
        usuario_id,
        mov_archivo,
        mov_visible_publico
    ) VALUES (
        IDDO,
        ORIG,
        IF(DEST = 0 OR DEST IS NULL, NULL, DEST),  -- NULL si es FINALIZAR
        DESCRIP,
        ESTATUS,  -- PENDIENTE (si deriva) o FINALIZADO (si finaliza)
        NOW(),
        IDUSU,
        RUTA,
        VISIBLE_PUBLICO
    );

    -- PASO 3: Actualizar el documento
    UPDATE documento SET
        doc_estatus = IF(ACCION = 'FINALIZAR', 'FINALIZADO', doc_estatus)
    WHERE documento_id = IDDO;
END$$

DELIMITER ;

SELECT '✅ SP_REGISTRAR_TRAMITE_DERIVAR corregido exitosamente' AS Resultado;

-- ========================================
-- CORREGIR DATOS EXISTENTES (Movimiento 1)
-- ========================================

-- Actualizar el movimiento 1 que quedó vacío
UPDATE movimiento
SET mov_estatus = 'DERIVADO'
WHERE movimiento_id = 1
AND documento_id = '001-2025'
AND (mov_estatus = '' OR mov_estatus IS NULL);

SELECT '✅ Movimiento 1 corregido a DERIVADO' AS Resultado;

-- ========================================
-- VERIFICACIÓN
-- ========================================

SELECT 'Estado actual de los movimientos:' AS Info;

SELECT
    movimiento_id,
    documento_id,
    area_origen_id,
    area_destino_id,
    mov_estatus,
    mov_fecharegistro,
    CASE
        WHEN movimiento_id = (
            SELECT MAX(m2.movimiento_id)
            FROM movimiento m2
            WHERE m2.documento_id = movimiento.documento_id
        ) THEN '✅ ÚLTIMO'
        ELSE '⬜ ANTERIOR'
    END AS posicion
FROM movimiento
WHERE documento_id = '001-2025'
ORDER BY movimiento_id;

-- ========================================
-- INSTRUCCIONES
-- ========================================
-- 1. Ejecuta este script completo en phpMyAdmin
-- 2. El SP está corregido para futuros movimientos
-- 3. El movimiento 1 ya fue corregido a DERIVADO
-- 4. Ahora prueba nuevamente:
--    - Informática debe ver el trámite en "Trámites Derivados" ✅
--    - Informática debe ver el trámite en "Trámites Recibidos" ✅
--    - Gerencia NO debe verlo en "Trámites Recibidos" ✅
